#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dictionary.h"

#define MAX_CMD_LEN 128

// A helper function to spell check a specific file
// 'file_name': Name of the file to spell check
// 'dict': A dictionary containing correct words
// returns -1 if failed and 0 if successful
int spell_check_file(const char *file_name, const dictionary_t *dict) {
    FILE* check_file = fopen(file_name, "r");
    if (check_file == NULL){
        printf("Spell check failed\n");
        return -1;
    }
    char word_temp[128] ={};
    char char_temp = '\0';
    int stringlen = 0;
    // loops through file reading one character at a time
    while(fscanf(check_file,"%c", &char_temp) != EOF){
        if(isalpha(char_temp) != 0){
            // if it is alphabet character
            word_temp[stringlen] = char_temp;
            stringlen++;
        }else if(stringlen > 0){
            // if a word is in word temp, and a space is detected, 
            // it is compared and printed to terminal
            // adds null terminator to end of string
            
            word_temp[stringlen] = '\0';
            if (dict_find(dict, word_temp) == 1){
                // if the word was found, print it as normal
                printf("%s", word_temp);
            }else{
                // if word not found, print it with a [X] after the word
                printf("%s[X]", word_temp);
            }
            // resets word_temp
            stringlen = 0;
            // prints same space character that was read to 
            // maintain spacing and newlines
            printf("%c", char_temp);
        }else{
            // if its a space character twice in a row, print it
            printf("%c", char_temp);
        }
    }
    fclose(check_file);
    // entire file is read
    if(stringlen > 0){
        // if there is still a word in the array at file close
        // adds null terminator to end of string
        word_temp[stringlen] = '\0';
        if (dict_find(dict, word_temp) == 1){
            // if the word was found, print it as normal
            printf("%s", word_temp);
        }else{
            // if word not found, print it with a [X] after the word
            printf("%s[X]", word_temp);
        }
        // resets word_temp
        stringlen = 0;
    }
    return 0;
}

/*
 * This is in general *very* similar to the list_main file seen in lab
 */
int main(int argc, char **argv) {
    dictionary_t *dict = NULL;
    if (argc > 1){
        // loads dictionary from file based on arguments
        dict = read_dict_from_text_file(argv[1]);
        if (dict == NULL) {
            // if failed, returns and closes
            printf("Failed to read dictionary from text file\n");
            return -1;
        }
        // prints read dict success message
        printf("Dictionary successfully read from text file\n");

        // checks if a file was provided to spellcheck
        if (argc > 2){
            // if it was, the file is spellchecked and closes application
            // if it failes
            if (spell_check_file(argv[2], dict) == -1){
                dict_free(dict);
                return -1;
            }else{
                // spell check was successful
                dict_free(dict);
                return 0;
            }
        }
    }else{
        // loads empty dictionary if no dict file name is provided
        dict = malloc(sizeof(dictionary_t));
        dict->size = 0;
        dict->root = NULL;
    }
    
    char cmd[MAX_CMD_LEN];
    char cmd_arg[MAX_CMD_LEN];

    printf("CSCI 2021 Spell Check System\n");
    printf("Commands:\n");
    printf("  add <word>:              adds a new word to dictionary\n");
    printf("  lookup <word>:           searches for a word\n");
    printf("  print:                   shows all words currently in the dictionary\n");
    printf("  load <file_name>:        reads in dictionary from a file\n");
    printf("  save <file_name>:        writes dictionary to a file\n");
    printf("  check <file_name>: spell checks the specified file\n");
    printf("  exit:                    exits the program\n");

    while (1) {
        printf("spell_check> ");
        if (scanf("%s", cmd) == EOF) {
            printf("\n");
            break;
        }

        if (strcmp("exit", cmd) == 0) {
            // exit command
            break;
        }else if (strcmp("add", cmd) == 0) {
            // add command
            // gets word to add
            scanf("%s", cmd_arg);
            dict_insert(dict, cmd_arg);
        }else if(strcmp("lookup", cmd) == 0){
            // lookup command
            // gets word to lookup
            scanf("%s", cmd_arg);
            if (dict_find(dict, cmd_arg) == 1){
                // word was found in dict
                printf("'%s' present in dictionary\n", cmd_arg);
            }else{
                // word not found in dict
                printf("'%s' not found\n", cmd_arg);
            }
        }else if(strcmp("print", cmd) == 0){
            // print dict cmd
            dict_print(dict);
        }else if(strcmp("load", cmd) == 0){
            // loads dict from file
            // gets file name to load from
            scanf("%s", cmd_arg);
            dictionary_t* tmp_dict = read_dict_from_text_file(cmd_arg);
            if (tmp_dict == NULL){
                // couldnt read and load file
                printf("Failed to read dictionary from text file\n");
            }else{
                // successfully loaded dict from file
                dict_free(dict);
                dict = tmp_dict;
                printf("Dictionary successfully read from text file\n");
            }
        }else if(strcmp("save", cmd) == 0){
            // gets filename to write to
            scanf("%s", cmd_arg);
            // if write failed
            if (write_dict_to_text_file(dict, cmd_arg) == -1){
                printf("Failed to write dictionary to text file\n");
            }else{
                // print successful message
                printf("Dictionary successfully written to text file\n");
            }
        }else if(strcmp("check", cmd) == 0){
            // gets filename to check spelling of
            scanf("%s", cmd_arg);
            spell_check_file(cmd_arg, dict);
            
        }
        else {
            printf("Unknown command %s\n", cmd);
        }
    }

    dict_free(dict);
    return 0;
}
